<?php  
require '../vendor/autoload.php'; // MongoDB PHP library

header("Content-Type: application/json");

$client     = new MongoDB\Client("mongodb://localhost:27017");
$collection = $client->clicktracker->wp_rotator_link_hit_logs;

// --- Access Key Validation ---
$access_key = $_POST['access_key'] ?? '';
if ($access_key !== 'qwerty403') {
    echo json_encode(["status" => "error", "message" => "Invalid access key"]);
    exit;
}

$link_id = $_POST['link_id'] ?? '';
if (empty($link_id)) {
    echo json_encode(["status" => "error", "message" => "Missing link_id"]);
    exit;
}

// --- Date Range ---
$start_date = date('Y-m-d', strtotime('-6 days')) . ' 00:00:00';
$end_date   = date('Y-m-d') . ' 23:59:59';

if (!empty($_POST['start_date']) && !empty($_POST['end_date'])) {
    $start_date = $_POST['start_date'] . ' 00:00:00';
    $end_date   = $_POST['end_date'] . ' 23:59:59';
}

// Convert to MongoDB UTCDateTime
$start = new MongoDB\BSON\UTCDateTime(strtotime($start_date) * 1000);
$end   = new MongoDB\BSON\UTCDateTime(strtotime($end_date) * 1000);

// --- Base Filter ---
$baseFilter = [
    'link_id'  => (string)$link_id,
    'hit_time' => [
        '$gte' => $start,
        '$lte' => $end
    ]
];

/*$linkDoc = $linksCollection->findOne(
    ['id' => (string)$link_id],
    ['projection' => ['lname' => 1]]
);
$link_name = $linkDoc['lname'] ?? '';*/

// --- Pagination ---
$items_per_page = 10;
$paged = isset($_POST['paged']) ? max(1, (int)$_POST['paged']) : 1;
$skip  = ($paged - 1) * $items_per_page;

// --- Counts ---
$total_clicks       = $collection->countDocuments($baseFilter);
$total_pages        = ceil($total_clicks / $items_per_page);

$unique_clicks      = $collection->distinct("ip", $baseFilter);
$unique_clicks_count = count($unique_clicks);

$fake_clicks_count  = $collection->countDocuments(array_merge($baseFilter, ['fake_click_status' => 1]));
$mobile_count       = $collection->countDocuments(array_merge($baseFilter, ['deviceType' => 'Mobile']));
$desktop_count      = $collection->countDocuments(array_merge($baseFilter, ['deviceType' => 'Desktop']));

// --- Fetch Paginated Results ---
$cursor = $collection->find(
    $baseFilter,
    [
        'skip'  => $skip,
        'limit' => $items_per_page,
        'sort'  => ['hit_time' => -1] // DESC order
    ]
);

$rows = [];
foreach ($cursor as $doc) {
    $rows[] = [
        'fake_click_status' => isset($doc['fake_click_status']) ? (int)$doc['fake_click_status'] : null,
        'ip'                => $doc['ip'] ?? '',
        'deviceType'        => $doc['deviceType'] ?? '',
        'hit_time'          => isset($doc['hit_time']) ? $doc['hit_time']->toDateTime()->format('Y-m-d H:i:s') : ''
    ];
}

$pipeline = [
    [
        '$match' => $baseFilter   // filter by link_id + date range
    ],
    [
        '$facet' => [
            "countryStats" => [
                [
                    '$group' => [
                        '_id'   => '$country',
                        'count' => ['$sum' => 1]
                    ]
                ],
                [
                    '$sort' => ['count' => -1]
                ]
            ],
            "totalStats" => [
                [
                    '$group' => [
                        '_id' => null,
                        'total' => ['$sum' => 1]
                    ]
                ]
            ]
        ]
    ],
    [
        '$unwind' => '$totalStats'
    ],
    [
        '$project' => [
            "countryStats" => [
                '$map' => [
                    'input' => '$countryStats',
                    'as'    => 'c',
                    'in'    => [
                        'country'    => '$$c._id',
                        'count'      => '$$c.count',
                        'percentage' => [
                            '$round' => [
                                [
                                    '$multiply' => [
                                        ['$divide' => ['$$c.count', '$totalStats.total']],
                                        100
                                    ]
                                ],
                                2
                            ]
                        ]
                    ]
                ]
            ]
        ]
    ]
];

$countryResults = $collection->aggregate($pipeline)->toArray();

$countryRows = [];
$countries = [];
$percentages = [];

if (!empty($countryResults[0]['countryStats'])) {
    foreach ($countryResults[0]['countryStats'] as $row) {
        $countryRows[] = $row;
        $countries[]   = $row['country'];
        $percentages[] = $row['percentage'];
    }
}

// --- Response ---
echo json_encode([
    "status"         => "success",
     //"link_name"      => $link_name,  
    "total_clicks"   => $total_clicks,
    "unique_clicks"  => $unique_clicks_count,
    "fake_clicks"    => $fake_clicks_count,
    "mobile_clicks"  => $mobile_count,
    "desktop_clicks" => $desktop_count,
    "total_pages"    => $total_pages,
    "paged"          => $paged,
    "data"           => $rows,
    "country_data"   => $countryRows,
    "countries"      => $countries,
    "percentages"    => $percentages
], JSON_PRETTY_PRINT);

